/****************************************************************************
 *   $Id:: pmu.c 3635 2010-06-02 00:31:46Z usb00423                         $
 *   Project: NXP lpc11U6x PMU example
 *
 *   Description:
 *     This file contains PMU code example which include PMU 
 *     initialization, PMU interrupt handler, and APIs for PMU
 *     reading.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
#include "lpc11U6x.h"			/* lpc11U6x Peripheral Registers */
#include "type.h"
#include "lpc11U6x_gpio.h"
#include "lpc11U6x_pmu.h"

/*****************************************************************************
** Function name:		PMU_Init
**
** Descriptions:		Initialize PMU and setup wakeup source.
**						For Sleep and deepsleep, any of the I/O pins can be 
**						used as the wakeup source.
**						For Deep Powerdown, only pin P1.4 can be used as 
**						wakeup source from deep powerdown. 
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void PMU_Init( void )
{
  uint32_t bitPattern[3];
  uint32_t eventPattern[3];

#if __RTCINT
  /* rtc int as wakeup source */
  LPC_SYSCON->STARTERP1 = STARTERP1_RTCINT;
#else
#if __WWDT_BODINT
  /* WWDT_BOD int as wakeup source */
  LPC_SYSCON->STARTERP1 = STARTERP1_WWDT_BODINT;
#else
#if __USART1_4
	/* USART1_4 as wakeup source */
  LPC_SYSCON->STARTERP1 = STARTERP1_USART1_4;
#else
#if __USART2_3
  /* USART2_3 as wakeup source */
  LPC_SYSCON->STARTERP1 = STARTERP1_USART2_3;
#else
#if 0
  bitPattern[0] = 0x00000020;		/* PIO0_5 enabled */
  bitPattern[1] = 0x00000004;		/* PIO1_2 enabled */
  bitPattern[2] = 0x00000004;		/* PIO2_2 enabled */
  eventPattern[0] = 0x00000020;		/* PIO0_5 rising edge */
  eventPattern[1] = 0x00000000;		/* PIO1_2 falling edge */
  eventPattern[2] = 0x00000000;		/* PIO2_2 falling edge */
  /* GINT0_INT sources OR together */
  GPIOSetGroupedInterrupt( GROUP0, &bitPattern[0], 0, 0, &eventPattern[0] );
  /* GINT0_INT as wakeup source */
  LPC_SYSCON->STARTERP1 = STARTERP1_GROUP0INT;
#endif

#if 0
  bitPattern[0] = 0x00000020;		/* PIO0_5 enabled */
  bitPattern[1] = 0x00000004;		/* PIO1_2 enabled */
  bitPattern[2] = 0x00000004;		/* PIO2_2 enabled */
  eventPattern[0] = 0x00000020;		/* PIO0_5 rising edge */
  eventPattern[1] = 0x00000000;		/* PIO1_2 falling edge */
  eventPattern[2] = 0x00000000;		/* PIO2_2 falling edge */
  /* GINT0_INT sources AND together */
  GPIOSetGroupedInterrupt( GROUP1, &bitPattern[0], 1, 0, &eventPattern[0] );
  /* GINT0_INT as wakeup source */
  LPC_SYSCON->STARTERP1 = STARTERP1_GROUP1INT;
#endif

#if 0
  /* PIN_INT0 source */
  GPIOSetPinInterrupt( CHANNEL0, PORT0, 16, 0, 0 );
  /* PIN_INT0 as wakeup source */
  LPC_SYSCON->STARTERP0 = STARTERP0_PINT0;
#endif

#if 0
  /* PIN_INT1 source */
  GPIOSetPinInterrupt( CHANNEL1, PORT0, 17, 0, 0 );
  /* PIN_INT1 as wakeup source */
  LPC_SYSCON->STARTERP0 = STARTERP0_PINT1;
#endif

#if 0
  /* PIN_INT2 source */
  GPIOSetPinInterrupt( CHANNEL2, PORT0, 18, 0, 0 );
  /* PIN_INT2 as wakeup source */
  LPC_SYSCON->STARTERP0 = STARTERP0_PINT2;
#endif

#if 0
  /* PIN_INT3 source */
  GPIOSetPinInterrupt( CHANNEL3, PORT0, 19, 0, 0 );
  /* PIN_INT3 as wakeup source */
  LPC_SYSCON->STARTERP0 = STARTERP0_PINT3;
#endif

#if 0
  /* PIN_INT4 source */
  GPIOSetPinInterrupt( CHANNEL4, PORT0, 20, 0, 0 );
  /* PIN_INT4 as wakeup source */
  LPC_SYSCON->STARTERP0 = STARTERP0_PINT4;
#endif

#if 0
  /* PIN_INT5 source */
  GPIOSetPinInterrupt( CHANNEL5, PORT0, 21, 0, 0 );
  /* PIN_INT5 as wakeup source */
  LPC_SYSCON->STARTERP0 = STARTERP0_PINT5;
#endif

#if 0
  /* PIN_INT6 source */
  GPIOSetPinInterrupt( CHANNEL6, PORT0, 22, 0, 0 );
  /* PIN_INT6 as wakeup source */
  LPC_SYSCON->STARTERP0 = STARTERP0_PINT6;
#endif

#if 1
  /* PIN_INT7 source */
  GPIOSetPinInterrupt( CHANNEL7, PORT0, 23, 0, 0 );
  /* PIN_INT7 as wakeup source */
  LPC_SYSCON->STARTERP0 = STARTERP0_PINT7;
#endif
#endif
#endif
#endif
#endif

  return;
}

/*****************************************************************************
** Function name:		PMU_Sleep
**
** Descriptions:		Put some of the peripheral in sleep mode.
**
** parameters:			SleepMode: 2 is power down, 1 is deep sleep, 0 is sleep, 
**						Sleep peripheral module(s)
** Returned value:		None
** 
*****************************************************************************/
void PMU_Sleep( uint32_t SleepMode, uint32_t SleepCtrl )
{
  LPC_SYSCON->PDAWAKECFG = LPC_SYSCON->PDRUNCFG;
#if 0
  LPC_SYSCON->PDRUNCFG |= (0x1 << 3);
#endif
#if 0
  LPC_SYSCON->PDAWAKECFG |= (0x1 << 3);
#endif
  LPC_SYSCON->PDSLEEPCFG = SleepCtrl;
  /* If normal sleep, not deep sleep, don't do anything to SCR reg. */
  switch ( SleepMode )
  {
	case MCU_POWER_DOWN:
	  LPC_PMU->PCON |= ((0x1<<5)|(0x1<<6));
	  SCB->SCR |= NVIC_LP_SLEEPDEEP;
#if 1
	  LPC_PMU->PCON |= 0x2;
#else
	  LPC_SYSCON->PDSLEEPCFG &= 0xFDFF;
#endif	
	  break;
	case MCU_DEEP_SLEEP:
	  SCB->SCR |= NVIC_LP_SLEEPDEEP;
#if 1
	  LPC_PMU->PCON |= 0x1;
#else
	  LPC_SYSCON->PDSLEEPCFG &= 0xEFF9;
#endif	
	  break;
	case MCU_SLEEP:
	default:
	  break;
  }
  __WFI();
  return;
}

/*****************************************************************************
** Function name:		PMU_DeepPowerDown
**
** Descriptions:		Some of the content should not be touched 
**						during the deep power down to wakeup process.
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void PMU_DeepPowerDown( void )
{
  uint32_t regVal;

  if ( (LPC_PMU->PCON & (0x1<<11)) != 0x0 )
  {
    /* Check deep power down bits. If deep power down mode is entered, 
    clear the PCON bits. */
    regVal = LPC_PMU->PCON;
    regVal |= (0x1<<11);
    LPC_PMU->PCON = regVal;

    if ( (LPC_PMU->GPREG0 != 0x12345678)||(LPC_PMU->GPREG1 != 0x87654321)
         ||(LPC_PMU->GPREG2 != 0x56781234)||(LPC_PMU->GPREG3 != 0x43218765) )
    {
      while (1);
    }
  }
  else
  {
	/* If in neither sleep nor deep power mode, enter deep power
	down mode now. */
    LPC_PMU->GPREG0 = 0x12345678;
    LPC_PMU->GPREG1 = 0x87654321;
    LPC_PMU->GPREG2 = 0x56781234;
    LPC_PMU->GPREG3 = 0x43218765;
    SCB->SCR |= NVIC_LP_SLEEPDEEP;
    LPC_PMU->PCON = 0x3;
    __WFI();
  }
  return;
}

/******************************************************************************
**                            End Of File
******************************************************************************/
